#include "functor.h"
#include "substitution.h"
#include "mode.h"
#include "tokenizer.h"
#include "unification.h"


namespace lp {
	using namespace std;
	using namespace useful;

	Functor* make_functor_ptr(const string& s, const Dictionary& dic)
	{
		stringstream ss;
		ss << s;
		ss << " ."; // add termination symbol
		return stream_to_expr(ss,dic);
	}

	Functor make_functor(const string& s, const Dictionary& dic)
	{
		stringstream ss;
		ss << s;
		ss << " ."; // add termination symbol
		Functor* ptr = stream_to_expr(ss,dic);
		if (!ptr) return Functor();
		Functor f = std::move(*ptr);
		delete ptr;
		return f;
	}

	Functor* make_general(id_type s, Functor::arity_type a)
	{
		Functor* f = new Functor(s);
		f->arg_reserve(a);
		while (a-- > 0) f->steal(new Functor(Functor::unique_index()));
		return f;
	}

	//bool is_subset(const Functor& f, const Functor& g, bool ordered)
	//{
	//	if (f.head()) {
	//		if (!g.head()) return false;
	//		if (*f.head() != *g.head()) return false;
	//	} else if (g.head()) {
	//		return false;
	//	} // else: either they both have the same head, or no heads
	//	auto i = f.body_begin();
	//	auto j = g.body_begin();
	//	for (auto i = f.body_begin(); i != f.body_end(); ++i) {
	//		if (!ordered) j = g.body_begin();
	//		for ( ; j != g.body_end(); ++j) {
	//			if (**i == **j) break;
	//		}
	//		if (j == g.body_end()) return false; // *i not found
	//		++j; // move j forward one step from found
	//	}
	//	return true;
	//}

	//bool is_renamed_subset(const Functor& f, const Functor& g, bool ordered)
	//{
	//	map<id_type,id_type> rm, im;
	//	if (!f.head()->is_renaming(g.head(),rm,im)) return false;
	//	for (auto i = f.body_begin(); i != f.body_end(); ++i) {
	//		auto j = ordered ? std::next(i) : g.body_begin();
	//		for ( ; j != g.body_end(); ++j) {
	// BUG: if renaming fails, it doesn't undo changes to rm/im
	//			if (i->is_renaming(*j,rm,im)) break;
	//		}
	//		if (j == g.body_end()) return false; // *i not found
	//	}
	//	return true;
	//}

	//******** Member Functions ********//

	
	
	Functor& Functor::operator=(const Functor& f)
	{
		if (this != &f) {
			// Note: this does not detach *this functor from its parent
			for_each(argv.begin(),argv.end(),[](Functor* c){delete c;});
			id(f.id());
			argv.resize(f.argv.size());
			for (arity_type k = 0; k < f.argv.size(); ++k) {
				argv[k] = f.argv[k]->copy();
				argv[k]->parent() = this;
			}
		}
		return *this;
	}

	
	Functor& Functor::operator=(Functor&& f)
	{
		if (this != &f) {
			// Note: this does not detach *this functor from its parent
			for_each(argv.begin(),argv.end(),[](Functor* c){delete c;});
			id(f.id());
			argv = std::move(f.argv);
			for_each(argv.begin(),argv.end(),[&](Functor* c){c->parent() = this;});
		}
		return *this;
	}

		
	bool Functor::is_variable(const Subst& subs) const
	{
		if (!is_variable()) return false;
		const Functor* b = subs.get(id());
		if (!b) return true;
		// Has binding, is binding a variable?
		return b->is_variable();
	}

	
	bool Functor::is_constant(const Subst& s) const
	{
		if (is_constant()) return true;
		if (is_variable()) {
			const Functor* f = s.get(id());
			if (!f) return false;
			return f->is_constant();
		} else return false;
	}

	bool Functor::is_numeric() const 
	{
		if (!argv.empty()) return false;
		try {
			fmap.get_numeric(id());
			return true;
		} catch (non_numeric) { return false; }
	}

	bool Functor::is_numeric(const Subst& s) const
	{
		if (is_numeric()) return true;
		if (is_variable()) {
			const Functor* f = s.get(id());
			if (!f) return false;
			return f->is_numeric();
		} else return false;
	}

	bool Functor::is_int() const 
	{
		if (!argv.empty()) return false;
		try {
			fmap.get_int(id());
			return true;
		} catch (non_numeric) { return false; }
	}

	bool Functor::is_int(const Subst& s) const
	{
		if (is_int()) return true;
		if (is_variable()) {
			const Functor* f = s.get(id());
			if (!f) return false;
			return f->is_int();
		} else return false;
	}

	bool Functor::is_double() const 
	{
		if (!argv.empty()) return false;
		try {
			fmap.get_float(id());
			return true;
		} catch (non_numeric) { return false; }
	}

	bool Functor::is_double(const Subst& s) const
	{
		if (is_double()) return true;
		if (is_variable()) {
			const Functor* f = s.get(id());
			if (!f) return false;
			return f->is_double();
		} else return false;
	}

	long long Functor::to_int(const Subst& subs) const
	{
		if (is_variable()) {
			const Functor* f = subs.get(id());
			if (!f) throw non_numeric();
			return f->to_int();
		} else {
			return to_int();
		}
	}

	double Functor::to_double(const Subst& subs) const
	{
		if (is_variable()) {
			const Functor* f = subs.get(id());
			if (!f) throw non_numeric();
			return f->to_double();
		} else {
			return to_double();
		}
	}

	const char* Functor::to_string(const Subst& subs) const
	{
		if (is_variable()) {
			const Functor* f = subs.get(id());
			if (!f) throw conversion_error();
			return f->to_string();
		} else {
			return to_string();
		}
	}


	long long Functor::to_unsigned(const Subst& subs) const
	{
		if (is_variable()) {
			const Functor* f = subs.get(id());
			if (!f) throw non_numeric();
			return f->to_unsigned();
		} else {
			return to_unsigned();
		}
	}

	data Functor::to_numeric(const Subst& subs) const
	{
		if (is_variable()) {
			const Functor* f = subs.get(id());
			if (!f) throw non_numeric();
			return f->to_numeric();
		} else {
			return to_numeric();
		}
	}


	bool Functor::is_function(const Subst& s) const
	{
		if (is_function()) return true;
		if (is_variable()) {
			const Functor* f = s.get(id());
			if (!f) return false;
			return f->is_function();
		}
		return false;
	}


	bool Functor::check_parents() const
	{
		// Depth first check that all children are connected to parent
		for (auto i = arg_begin(); i != arg_end(); ++i) {
			if ((*i)->parent() != this) {
				DEBUG_WARNING(cerr << "Error: " << **i << "does not point to right parent\n");
				return false;
			}
			// Traverse down
			if (!(*i)->check_parents()) return false;
		}
		return true;
	}
	
	void Functor::steal_all(Functor* p)
	{
		argv.reserve(argv.size() + p->argv.size());
		for_each(p->argv.begin(),p->argv.end(),[&](Functor* c){
			c->parent() = this;
			this->argv.push_back(c);
		});
		p->argv.clear();
	}

	
	bool Functor::operator==(const Functor& f) const
	{
		if (id() != f.id() || arity() != f.arity()) return false;
		for (arity_type i = 0; i < arity(); ++i) {
			if (*arg(i) != *f.arg(i)) return false;
		}
		return true;
	}

	
	int Functor::compare(const Functor* f) const
	{
		// Arity
		if (arity() < f->arity()) return -1;
		else if (arity() > f->arity()) return 1;
		// Symbols
		if (id() < f->id()) return -1;
		if (id() > f->id()) return 1;
		// Recursive (f/n = g/m => same arity)
		int d;
		for (Functor::arity_type k = 0; k < arity(); ++k) {
			d = arg(k)->compare(f->arg(k));
			if (d != 0) return d; // else continue
		}
		return 0; // everything equal
	}

	
	int Functor::compare(const Functor* f, const Subst& subs) const
	{
		// Arity
		if (arity() < f->arity()) return -1;
		else if (arity() > f->arity()) return 1;
		// Symbols
		const Functor* tsub = this;
		const Functor* fsub = f;
		if (this->is_variable() && (tsub = subs.get(this->id())) == nullptr) tsub = this;
		if (f->is_variable() && (fsub = subs.get(f->id())) == nullptr) fsub = f;
		if (tsub->id() < fsub->id()) return -1;
		if (tsub->id() > fsub->id()) return 1;
		// Recursive (f/n = g/m => same arity)
		int d;
		for (Functor::arity_type k = 0; k < arity(); ++k) {
			d = arg(k)->compare(f->arg(k),subs);
			if (d != 0) return d; // else continue
		}
		return 0; // everything equal
	}


	int Functor::compare_prolog(const Functor* g) const
	{
		// Variables < Numbers < Atoms < Compound

		// Variables?
		if (is_variable()) {
			if (g->is_variable()) {
				// compare IDs
				if (id() < g->id()) return -1;
				if (id() > g->id()) return 1;
				return 0;
			} else {
				return -1;
			}
		} else if (g->is_variable()) {
			return 1;
		}
		
		const data d1 = Functor::get_data(id());
		const data d2 = Functor::get_data(g->id());

		assert(!d1.is_nil() && !d2.is_nil());

		switch (d1.type()) {
		case data::int_tag: case data::float_tag:
			switch (d2.type()) {
			case data::int_tag: case data::float_tag: 
				return numeric_compare(eval(),g->eval());
			default: return -1;
			}
		default:
			switch (d2.type()) {
			case data::int_tag: case data::float_tag: return 1;
			}
		}
		assert(d1.is_atom() && d2.is_atom());

		// atom or compound
		// Arity
		if (arity() < g->arity()) return -1;
		else if (arity() > g->arity()) return 1;
		// Symbols
		int ord = strcmp(d1.get_atom(),d2.get_atom());
		if (ord < 0) return -1;
		if (ord > 0) return 1;
		// Recursive (f/n = g/m => same arity)
		for (Functor::arity_type k = 0; k < arity(); ++k) {
			ord = arg(k)->compare_prolog(g->arg(k));
			if (ord < 0) return -1;
			if (ord > 0) return 1;
		}
		return 0; // everything equal
	}

	
	data Functor::eval(const Subst& subs) const
	{
		// Prolog has unary and binary arithmetic operators
		if (is_leaf()) {
			if (is_numeric()) { 
				return this->to_numeric();
			} else if (is_variable()) {
				const Functor* fp = subs.get(id());
				if (fp) {
					return fp->eval(subs);
				} else {
					throw eval_error(string("variable ") + get_data(id()).get_atom() + " not instantiated in numeric eval");
				}
			} else {
				throw eval_error("atoms not allowed in numerical evaluation");
			}
		} else if (arity() == 1) {
			if (id() != minus_id) throw eval_error("unrecognized unary arithmetic operator");
			data ans = arg_first()->eval(subs);
			switch (ans.type()) {
			case data::int_tag: ans.set_int(-ans.get_int()); return ans;
			case data::float_tag: ans.set_float(-ans.get_float()); return ans;
			}
		} else if (arity() == 2) {
			data lhs = arg_first()->eval(subs);
			data rhs = arg_last()->eval(subs);
			// 4 combinations of integer/double
			if (lhs.type() == data::int_tag) {
				if (rhs.type() == data::int_tag) {
					switch (id()) {
					case plus_id: return data(lhs.get_int() + rhs.get_int());
					case minus_id: return data(lhs.get_int() - rhs.get_int());
					case mul_id: return data(lhs.get_int() * rhs.get_int());
					case div_id: { // convert to double if we have to
						const double dd = double(lhs.get_int()) / rhs.get_int();
						const long long ii = long long(dd);
						return (ii == dd) ? data(ii) : data(dd);
								 }
					case star2_id:
					case pow_id: { // convert to int if we have to
						const double dd = pow(double(lhs.get_int()),double(rhs.get_int()));
						const long long ii = int(dd);
						return (ii == dd) ? data(ii) : data(dd);
								 }
					default: throw eval_error("non-arithmetic functor");
					}
				} else {
					assert(rhs.is_double()); // eval for rhs must be float
					switch (id()) {
					case plus_id: return data(lhs.get_int() + rhs.get_float());
					case minus_id: return data(lhs.get_int() - rhs.get_float());
					case mul_id: return data(lhs.get_int() * rhs.get_float());
					case div_id: return data(lhs.get_int() / rhs.get_float());
						// for pow_id, we convert to double
					case pow_id: return data(std::pow(double(lhs.get_int()),double(rhs.get_float())));
					default: throw eval_error("non-arithmetic functor");
					}
				}
			} else {
				assert(lhs.is_double()); // eval for lhs must be double
				if (rhs.type() == data::int_tag) {
					switch (id()) {
					case plus_id: return data(lhs.get_float() + rhs.get_int());
					case minus_id: return data(lhs.get_float() - rhs.get_int());
					case mul_id: return data(lhs.get_float() * rhs.get_int());
					case div_id: return data(lhs.get_float() / rhs.get_int());
						// for pow_id, we convert to double
					case pow_id: return data(std::pow(double(lhs.get_float()),double(rhs.get_int())));
					default: throw eval_error("non-arithmetic functor");
					}
				} else {
					assert(rhs.is_double()); // eval for rhs must be double
					switch (id()) {
					case plus_id: return data(lhs.get_float() + rhs.get_float());
					case minus_id: return data(lhs.get_float() - rhs.get_float());
					case mul_id: return data(lhs.get_float() * rhs.get_float());
					case div_id: return data(lhs.get_float() / rhs.get_float());
						// for pow_id, we convert to double
					case pow_id: return data(std::pow(double(lhs.get_float()),double(rhs.get_float())));
					default: throw eval_error("non-arithmetic functor");
					}
				} 
			}
		}
		throw eval_error("could not evaluate expression");
	}



	bool Functor::is_renaming(
		const Functor* f,
		map<id_type,id_type>& vs,
		map<id_type,id_type>& rvs) const
	{
		// Ignore order of body literals
		if (is_variable()) {
			if (!f->is_variable()) return false;
			else {
				auto i = vs.find(id());
				if (i == vs.end()) {
					// first we must check that the subst will still be one-to-one
					// e.g. p(X,Y) is not a renaming of p(Z,Z)
					if (rvs.find(f->id()) != rvs.end()) return false;
					// new variable substitution
					vs[id()] = f->id();
					rvs[f->id()] = id();
					return true;
				} else {
					// already has substitution, check that it matches
					return i->second == f->id();
				}
			}
		} else if (f->is_variable()) {
			return false;
		} else {
			// not variable, compare as with ==
			if (id() != f->id() || arity() != f->arity()) return false;
			for (arity_type i = 0; i < arity(); ++i) {
				if (!arg(i)->is_renaming(f->arg(i),vs,rvs)) return false;
			}
			return true;
		}
	}
	
	Functor* Functor::copy() const
	{
		// Non-recursive implementation uses a std::map
		// Enables expressions with tens of thousands of chained nodes (lists)
		std::map<const Functor*,Functor*> fmap; // *this -> new
		Functor* root = new Functor(begin()->id());
		fmap.insert(std::make_pair(&*begin(),root));
		// Note: pre-order traversal ensures parents are created before children
		std::for_each(++begin(),end(),[&](const Functor& n){
			Functor* tmp = new Functor(n.id());
			assert(n.parent());
			auto par = fmap.find(n.parent());
			assert(par != fmap.end());
			// Parent already created, attach
			par->second->steal( tmp );
			fmap.insert(std::make_pair(&n,tmp));
		});
		return root;
	}
	
	Functor::arity_type Functor::body_size() const
	{
		if (is_empty_goal() || is_fact()) return 0;
		else {
			assert(is_goal() || is_rule());
			// size = 1 + number of ','/2
			arity_type size = 1;
			const Functor* ptr = arg_last();
			while (ptr->sign(sequence_id,2)) {
				++size;
				ptr = ptr->arg_last();
			}
			return size;
		}
	}

	
	void Functor::body_push_front(Functor* f)
	{
		assert(f);
		if (is_empty_goal()) {
			steal(f); // add f as only goal literal
		} else if (id() == if_id && (arity() == 1 || arity() == 2)) {
			Functor* subtree = arg_last(); // current first element
			remove_parent_pointer(subtree); // disconnect
			steal(new Functor(sequence_id));
			arg_last()->steal(f);
			// arg_last()->steal(subtree); <-- won't work since subtree has invalid parent
			subtree->parent() = arg_last();
			arg_last()->argv.push_back(subtree);
		} else {
			assert(is_fact());
			Functor* head = new Functor(id());
			head->steal_all(this);
			id(if_id);
			steal(head);
			steal(f);
		}
	}

	
	void Functor::body_push_back(Functor* f, Functor* last)
	{
		assert(f);
		if (is_empty_goal()) {
			steal(f); // add f as only goal literal
		} else if (id() == if_id && (arity() == 1 || arity() == 2)) {
			Functor* ptr = (last == nullptr ? this : last->parent());
			while (ptr->arg_last()->sign(sequence_id,2)) ptr = ptr->arg_last();
			// ptr is now standing above last element (at :- if singleton, ,/2 otherwise)
			Functor* subtree = ptr->arg_last(); // current last element
			remove_parent_pointer(subtree); // disconnect last element
			ptr->steal(new Functor(sequence_id));
			ptr = ptr->arg_last(); // move down to new ,/2
			// ptr->steal(subtree); <-- this won't work since subtree now has false parent
			subtree->parent() = ptr;
			ptr->argv.push_back(subtree);
			ptr->steal(f);
		} else {
			assert(is_fact());
			// h => h :- b
			Functor* head = new Functor(id());
			head->steal_all(this);
			id(if_id);
			steal(head);
			steal(f);
		}
	}

	
	void Functor::body_insert(Functor* f, seq_iterator pos)
	{
		if (is_empty_goal()) {
			steal(f); // add f as only goal literal
			assert(top()->check_parents());
		} else if (id() == if_id && arity() <= 2) {
			// Rule or non-empty goal
			// Insert in front of pos
			if (pos == seq_iterator()) {
				body_push_back(f);
				assert(top()->check_parents());
			} else if (pos->parent()->sign(sequence_id,2)) {
				Functor* after;
				if (pos->parent()->arg_first() == &*pos)
					after = pos->parent(); // not last
				else 
					after = &*pos; // last
				// Create subtree ','(f,...) ; works for last element too
				Functor* above = after->parent();
				Functor* fseq = new Functor(sequence_id,f,after);
				above->steal(fseq);
				assert(top()->check_parents());
			} else {
				// begin(): insert at front
				assert(top()->check_parents());
				assert(pos->parent() == this && pos->parent()->arg_last() == &*pos);
				body_push_front(f);
				assert(top()->check_parents());
			}
		} else {
			assert(is_fact());
			Functor* head = new Functor(id());
			head->steal_all(this);
			id(if_id);
			steal(head);
			steal(f);
			assert(top()->check_parents());
		}
	}
	
	void Functor::head(Functor* h)
	{
		if (h == nullptr) {
			head_clear();
			return;
		}
		if (is_nonempty_goal()) {
			steal_front(h);
			return;
		}
		Functor* curr_head = (is_rule()) ? arg_first() : this;
		curr_head->id(h->id());
		curr_head->arg_clear();
		curr_head->steal_all(h);
		delete h;
	}
	

	void Functor::body(Functor* b)
	{
		if (b == nullptr) {
			// Clear body
			body_clear();
		} else if (is_empty_goal()) {
			steal(b);
		} else if (is_rule() || is_goal()) {
			// Rule or goal
			delete argv.back();
			argv.back() = b;
			b->parent() = this;
		} else {
			// Fact
			assert(is_fact());
			Functor* head = new Functor(id());
			head->steal_all(this);
			id(if_id);
			assert(is_leaf());
			steal(head);
			steal(b);
		}
	}
	
	void Functor::body_pop_front()
	{
		assert(!is_empty_goal() && is_goal() || is_rule());

		if (arg_last()->sign(sequence_id,2)) {
			// h :- (b1,...)
			Functor tmp(0,arg_last()->arg_last()); // steal (b2,...)
			arg_pop_last(); // pop ,(b1)
			steal(tmp.arg_first());
		} else {
			// h :- b  =>  h
			arg_pop_last();
			if (!is_leaf()) { // has head, :- h(x,y)
				assert(arity() == 1);
				id(arg_first()->id()); // h(h(x,y))
				vec_type hargs = move(arg_first()->argv); // h(h)
				delete arg_first(); // h
				argv = move(hargs); // h(x,y)
				for_each(argv.begin(),argv.end(),[&](Functor* c){c->parent() = this;});
			}
		}
		assert(top()->check_parents());
	}

	
	void Functor::body_pop_back()
	{
		assert(is_nonempty_goal() || is_rule());

		if (arg_last()->sign(sequence_id,2)) {
			// h :- (b1,...,bk,bn) => h :- (b1,...,bk)
			Functor* ptr = arg_last();
			while (ptr->arg_last()->sign(sequence_id,2)) ptr = ptr->arg_last();
			ptr->arg_pop_last(); // erase last element
			// move up arg_first to current sequence location
			ptr->id(ptr->arg_first()->id());
			vec_type hargs = move(ptr->arg_first()->argv);
			delete ptr->arg_first();
			ptr->argv = move(hargs); // h(x,y)
			for_each(ptr->argv.begin(),ptr->argv.end(),[&](Functor* c){c->parent() = ptr;});
		} else {
			// h :- b  =>  h
			arg_pop_last();
			if (arity() == 1) { // has head, :- h(x,y)
				id(arg_first()->id()); // h(h(x,y))
				vec_type hargs = move(arg_first()->argv); // h(h)
				delete arg_first(); // h
				argv = move(hargs); // h(x,y)
				for_each(argv.begin(),argv.end(),[&](Functor* c){c->parent() = this;});
			}
		}
		assert(top()->check_parents());
	}

	
	Functor::seq_iterator Functor::body_erase(Functor::seq_iterator i)
	{
		assert(i != seq_iterator(nullptr));
		assert(!is_empty_goal() && is_goal() || is_rule());

		Functor* ptr = i->parent();
		assert(ptr);
		if (ptr->sign(sequence_id,2)) {
			if (ptr->arg_first() == &*i ) {
				// not last element
				// Disconnect tail from sequence
				Functor* tail = ptr->arg_last();
				remove_parent_pointer(tail); // disconnect
				// Erase element and its parent ','/2
				ptr = ptr->parent();
				// ptr is either ',' or :-, in both cases we came from ptr->arg_last()
				delete ptr->arg_last();
				ptr->argv.back() = tail;
				tail->parent() = ptr;
				assert(top()->check_parents());
				return seq_iterator(tail);
			} else {
				// last element
				assert(ptr->arg_last() == &*i);
				// Erase last element
				ptr->arg_pop_last();
				// Disconnect second to last element
				Functor* new_last = ptr->arg_first();
				remove_parent_pointer(new_last); // disconnect subtree
				// Move up and reconnect
				ptr = ptr->parent(); // ptr now points to ',' or :-
				delete ptr->arg_last();
				ptr->argv.back() = new_last;
				new_last->parent() = ptr;
				assert(top()->check_parents());
				return seq_iterator(nullptr);
			}
		} else {
			// singleton, erase it and remove :-
			assert(ptr->id() == if_id);
			ptr->arg_pop_last();
			if (ptr->arity() == 1) { // we have :- h(x,y)
				ptr->id(ptr->arg_first()->id()); // h(h(x,y))
				vec_type hargs = move(ptr->arg_first()->argv); // h(h)
				delete ptr->arg_first(); // h
				ptr->argv = move(hargs); // h(x,y)
				for_each(ptr->argv.begin(),ptr->argv.end(),[&](Functor* c){c->parent() = ptr;});
			}
			assert(top()->check_parents());
			return seq_iterator(nullptr);
		}
	}


	Functor* Functor::body_release(Functor::seq_iterator i)
	{
		assert(i != seq_iterator(nullptr));
		assert(!is_empty_goal() && is_goal() || is_rule());

		Functor* ptr = i->parent();
		assert(ptr);
		if (ptr->sign(sequence_id,2)) {
			if (ptr->arg_first() == &*i ) {
				// not last element
				// Disconnect tail from sequence
				Functor* tail = ptr->arg_last();
				remove_parent_pointer(tail); // disconnect
				// Disconnect target element
				Functor* target = ptr->arg_first();
				remove_parent_pointer(target);
				target->parent() = nullptr;
				// Move up and remove dangling ','
				ptr = ptr->parent();
				// ptr is now either ',' or :-, in both cases we came from ptr->arg_last()
				delete ptr->arg_last(); // erase dangling ','
				ptr->argv.back() = tail; // put back tail
				tail->parent() = ptr;
				assert(top()->check_parents());
				assert(target->parent() == nullptr);
				return target;
			} else {
				// last element
				assert(ptr->arg_last() == &*i);
				// Disconnect last element
				Functor* target = ptr->arg_last();
				remove_parent_pointer(target);
				target->parent() = nullptr;
				// Disconnect second to last element
				Functor* new_last = ptr->arg_first();
				remove_parent_pointer(new_last); // disconnect subtree
				// Move up and reconnect
				ptr = ptr->parent(); // ptr now points to ',' or :-
				delete ptr->arg_last();
				ptr->argv.back() = new_last;
				new_last->parent() = ptr;
				assert(top()->check_parents());
				assert(target->parent() == nullptr);
				return target;
			}
		} else {
			// singleton, erase it and remove :-
			assert(ptr->id() == if_id && ptr->arg_last() == &*i);
			Functor* target = ptr->arg_last();
			remove_parent_pointer(target);
			target->parent() = nullptr;
			if (ptr->arity() == 1) { // we have :- h(x,y)
				ptr->id(ptr->arg_first()->id()); // h(h(x,y))
				vec_type hargs = move(ptr->arg_first()->argv); // h(h)
				delete ptr->arg_first(); // h
				ptr->argv = move(hargs); // h(x,y)
				for_each(ptr->argv.begin(),ptr->argv.end(),[&](Functor* c){c->parent() = ptr;});
			}
			assert(top()->check_parents());
			assert(target->parent() == nullptr);
			return target;
		}
	}

	Functor* Functor::body_release()
	{
		if (is_rule()) {
			// Store body, disconnect it from this
			Functor* b = argv.back();
			argv.pop_back();
			b->parent() = nullptr;
			// Move up head: :- h => h
			id(arg_first()->id());
			vec_type hargs = move(arg_first()->argv);
			delete arg_first();
			argv = move(hargs);
			for_each(argv.begin(),argv.end(),[&](Functor* c){c->parent() = this;});
			return b;
		} else if (is_goal()) {
			Functor* b = argv.back();
			argv.pop_back();
			b->parent() = nullptr;
			return b;
		} else return nullptr;
	}

	bool Functor::is_tautology() const
	{
		const Functor* h = head();
		if (!h) return false;
		for (auto j = body_begin(); j != body_end(); ++j) {
			if (*j == *h) return true;
		}
		return false;
	}	

	bool Functor::is_ioconnected() const
	{
		if (!head()) return false;
		set<id_type> reg_vars;
		head()->variables(reg_vars);
		for (auto b = body_begin(); b != body_end(); ++b) {
			// At least one variable in b must be in reg_vars
			set<id_type> bvars;
			b->variables(bvars);
			bool is_connected = false;
			for (auto v = bvars.begin(); v != bvars.end(); ++v) {
				if (!is_connected && reg_vars.find(*v) != reg_vars.end()) {
					is_connected = true; // found one
				} else {
					// Register variable (update reg_vars)
					reg_vars.insert(*v);
				}
			}
			if (!is_connected) return false;
		}
		return true;
	}


	void Functor::flatten()
	{
		// Assumes clause has no side effects. This is locally impossible to decide (I think)
		// so it's up to the user to ensure that flatten() is only called when it's safe
		// For ILP, clauses with side effects give far more serious problems anyway

		auto i = body_begin();
		for ( ; i != body_end(); ) {
			if ( i->sign(equal2_id,2) ) {
				this->subst( i->arg_first()->id() , i->arg_last()->id() );
				// Erase this equality
				i = body_erase(i);
				continue;
			} else if ( i->sign(equal_id,2) ) {
				// Flatten X=anything and anything=X
				const Functor* left = i->arg_first();
				const Functor* right = i->arg_last();
				if (left->is_variable()) {
					this->subst( left->id(), right );
					i = body_erase(i);
					continue;
				} else if (right->is_variable()) {
					this->subst( right->id(), left );
					i = body_erase(i);
					continue;
				}
			}
			++i; // move to next
		}
	}


	const Functor* Functor::get(const position& p) const
	{
		const Functor* fp = this;
		for_each(p.begin(),p.end(),[&](position::index_type k){
			fp = fp->arg(k);
		});
		return fp;
	}


	void Functor::subst(id_type v, const Functor* f)
	{
		// Note: use post-order traversal when substituting
		std::for_each(post_begin(),post_end(),[&](Functor& n){
			if (n.is_variable() && n.id() == v) { 
				n = *f; 
			}
		});
	}

	void Functor::subst(id_type v, id_type w)
	{
		// Note: use post-order traversal when substituting
		std::for_each(post_begin(),post_end(),[&](Functor& n){
			if (n.is_variable() && n.id() == v) { 
				n.id(w);
			}
		});
	}

	void Functor::subst(const Subst& subs)
	{
		// Note: use post-order traversal when substituting
		std::for_each(post_begin(),post_end(),[&](Functor& n){
			const Functor* rep;
			if (n.is_variable() && (rep = subs.get(n.id())) != nullptr) {
				// replace this by root of rep
				n = *rep;
			}
		});
	}

	
	void Functor::replace_this(const Functor* trg)
	{
		id( trg->id() );
		arg_clear();
		// Copy all of trg's arguments
		argv.reserve(trg->arity());
		for_each(trg->arg_begin(),trg->arg_end(),[&](const Functor* f){
			argv.push_back(f->copy());
			argv.back()->parent() = this;
		});
	}

	
	void Functor::rename(map<id_type,id_type>& subs) 
	{
		map<id_type,id_type>::iterator it;
		for (Functor& c : *this) {
			if (c.is_variable()) {
				if ((it = subs.find(c.id())) == subs.end()) {
					const id_type vn = this->unique_index();
					subs.insert(map<id_type,id_type>::value_type(c.id(),vn));
					c.id(vn);
				} else {
					c.id( it->second ); // add new symbol
				}
			}
		}
	}

	
	void Functor::rename(map<id_type,id_type>& subs, map<id_type,id_type>& inv_subs) 
	{
		map<id_type,id_type>::iterator it;
		for (Functor& c : *this) {
			if (c.is_variable()) {
				if ((it = subs.find(c.id())) == subs.end()) {
					id_type vn = unique_index();
					subs.insert(make_pair(c.id(),vn));
					inv_subs.insert(make_pair(vn,c.id()));
					c.id(vn);
				} else c.id( it->second ); // add new symbol
			}
		}
	}

	
	void Functor::unrename(const map<id_type,id_type>& is) 
	{
		map<id_type,id_type>::const_iterator it;
		for (Functor& c : *this) {
			if (c.is_variable() && (it = is.find(c.id())) != is.end()) {
				c.id( it->second );
			}
		}
	}


	// Helper for unrename and format_variables
	void Functor::next_letter(string& s, const Functor& expr)
	{
		if (s.empty()) {
			s = 'A';
		} else {
			// propagate +1 to the left
			auto i = s.rbegin();
			for ( ; i != s.rend(); ++i) {
				if (*i == 'Z') *i = 'A'; // and continue propagation
				else {
					++(*i); // A -> B -> C -> ...
					break;
				}
			}
			// if propagation went to rend(), everything is A and we need one more
			if (i == s.rend()) s += 'A';
		}
		// Check if this character is free
		const id_type sid = Functor::index(s);
		auto found = std::find_if(expr.begin(),expr.end(),[sid](const Functor& n){ return n.sign(sid,0); });
		if (found != expr.end()) {
			return next_letter(s,expr);
		}
	}

	void Functor::print_this(ostream& os, string& vn, map<id_type,string>& m) const
	{
		if (is_reserved_variable() && pretty_print.is_int() && pretty_print.get_int() >= 1) {
			auto at = m.find(id());
			if (at == m.end()) {
				next_letter(vn,*this->top()); // get next letter
				at = m.insert(make_pair(id(),vn)).first;
			}
			os << at->second;
		} else {
			fmap.print(os,id());
		}
	}


	void Functor::print_prefix(ostream& os) const
	{
		fmap.print(os,id());

		if (arity() == 1) {
			os << "(";
			arg_first()->print_prefix(os);
			os << ")";
		} else if (arity() >= 2) {
			os << "(";
			arg_first()->print_prefix(os);
			for_each(++arg_begin(),arg_end(),[&os](const Functor* c){
				os << ",";
				c->print_prefix(os);
			});
			os << ")";
		}
	}



	void Functor::print_infix(ostream& os, const Dictionary& dic) const
	{
		map<id_type,string> vm;
		if (pretty_print.is_int() && pretty_print.get_int() >= 2) { // singleton mode
			// go through map and find all singleton variables
			typedef map<id_type,pair<const Functor*,bool> > umap;
			umap um;
			for (const Functor& f : *this) {
				if (f.is_variable()) {
					auto at = um.find(f.id());
					if (at == um.end()) {
						um.insert(make_pair(f.id(),make_pair(&f,true)));
					} else {
						at->second.second = false;
					}
				}
			}
			// grab all singletons
			std::for_each(um.begin(),um.end(),[&](const umap::value_type& p){
				if (p.second.second)
					vm.insert(make_pair(p.second.first->id(),"_"));
			});
		}
		string vn = "";
		return print_infix(os,vn,vm,dic);
	}

	
	void Functor::print_infix(
		ostream& os, 
		string& vn, // current renaming variable name
		map<id_type,string>& vm, // current renaming map
		const Dictionary& dic) const
	{
		if (is_seq()) {
			// print sequence as (1, 2, 3, ..., n)
			const bool body_lits = parent() && parent()->id() == if_id 
				&& parent()->arity() <= 2 && parent()->arg_last() == this;
			auto i = seq_begin();
			if (!body_lits) os << "(";
			i->print_infix(os,vn,vm,dic);
			for (++i ; i != seq_end(); ++i) {
				os << ", ";
				i->print_infix(os,vn,vm,dic);
			}
			if (!body_lits) os << ")";
		} else if (is_pair()) {
			// vector with at least two elements
			os << "[";
			const Functor* lp = arg_first();
			for (;;) {
				lp->print_infix(os,vn,vm,dic);
				lp = lp->parent()->arg_last();
				if (!lp->is_pair()) break; // last element
				lp = lp->arg_first();
				os << ",";
			}
			if (!lp->is_empty_list()) {
				os << "|";
				lp->print_infix(os,vn,vm,dic);
			}
			os << "]";
		} else if (is_infix_operator(dic)) {
			assert(arity() == 2);
			const int prec = precedence(dic);
			const Functor* larg = arg_first();
			const Functor* rarg = arg_last();
			bool print_p = (prec <= larg->precedence(dic));
			if (print_p) os << "(";
			larg->print_infix(os,vn,vm,dic);
			if (print_p) os << ")";
			os << " "; fmap.print(os,id()); os << " ";
			print_p = (prec <= rarg->precedence(dic));
			if (print_p) os << "(";
			rarg->print_infix(os,vn,vm,dic);
			if (print_p) os << ")";
		} else if (is_prefix_operator(dic)) {
			const bool print_p = (precedence(dic) <= arg_first()->precedence(dic));
			fmap.print(os,id());
			if (print_p) os << "(";
			else os << " ";
			arg_first()->print_infix(os,vn,vm,dic);
			if (print_p) os << ")";
		} else if (is_suffix_operator(dic)) {
			const bool print_p = (precedence(dic) <= arg_first()->precedence(dic));
			if (print_p) os << "(";
			arg_first()->print_infix(os,vn,vm,dic);
			if (print_p) os << ")";
			else os << " ";
			fmap.print(os,id());
		} else {
			// function, variable, or constant
			// print this as f(sequence)
			//cerr << "printing function... ";
			print_this(os,vn,vm);
			if (!is_leaf()) {
				os << "(";
				arg_first()->print_infix(os,vn,vm,dic);
				for_each(++arg_begin(),arg_end(),[&](const Functor* c){
					os << ",";
					c->print_infix(os,vn,vm,dic);
				});
				os << ")";
			}
		}
	}

	
	int Functor::precedence(const Dictionary& dic) const
	{
		if (is_leaf() || arity() > 2 || !is_function()) return 0;
		const char* atom;
		try {
			atom = fmap.get_atom(id());
		} catch (conversion_error) { return 0; }
		if (is_prefix_operator(dic))
			return dic.precedence(atom,Token::fx | Token::fy);
		if (is_suffix_operator(dic))
			return dic.precedence(atom,Token::xf | Token::yf);
		if (is_infix_operator(dic))
			return dic.precedence(atom,Token::xfy | Token::yfx | Token::xfx);
		return 0;
	}

	
	bool Functor::is_prefix_operator(const Dictionary& dic) const
	{
		const char* atom;
		try {
			atom = fmap.get_atom(id());
		} catch (conversion_error) { return false; }
		return is_function() && arity() == 1 && dic.has_prefix_op(atom);
	}
	

	bool Functor::is_infix_operator(const Dictionary& dic) const
	{
		const char* atom;
		try {
			atom = fmap.get_atom(id());
		} catch (conversion_error) { return false; }
		return is_function() && arity() == 2 && dic.has_infix_op(atom);
	}

	
	bool Functor::is_suffix_operator(const Dictionary& dic) const
	{
		const char* atom;
		try {
			atom = fmap.get_atom(id());
		} catch (conversion_error) { return false; }
		return is_function() && arity() == 1 && dic.has_suffix_op(atom);
	}



	// Position class
	Functor::position::position(const Functor* f)
	{
		// trace back to top, then reverse pos
		const Functor* p;
		while ( (p = f->parent()) != nullptr ) {
			// which child of p is f?
			for (arity_type k = 0; k < p->arity(); ++k) {
				if (p->arg(k) == f) {
					pos.push_back(k);
					break;
				}
			}
			f = p; // move up f one step
		}
		reverse(pos.begin(),pos.end());
	}



	unsigned long long hash_variant(const Functor& t, std::map<id_type,int>& vmap)
	{
		unsigned long long res = 0;
		unsigned long long s = 1;
		for (const Functor& n : t) {
			if (n.is_variable()) {
				auto at = vmap.find(n.id());
				int idx;
				if (at == vmap.end()) {
					idx = vmap.size()+1;
					vmap.insert(std::make_pair(n.id(),idx));
				} else {
					idx = at->second;
				}
				res += s * std::abs(idx);
			} else {
				res += s * std::abs(n.id()) * (1 + n.arity());
				s *= 31;
			}
		}
		return res;
	}


}


